<?php

/**
 * Copyright 2018 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
// [START sheets_quickstart]
require __DIR__ . '/vendor/autoload.php';

class QSM_Google_Sheets {

	private $client;
	private $token;
	private $spreadsheet;
	private $worksheet;

	const clientId = '204959849311-3588d9376t3ed8v37d2o3532thmucq5h.apps.googleusercontent.com';
	const clientSecret = '0bkRZsMiqB_Nusrnmm3dh7GN';

	public function __construct($spreadsheet = '', $worksheet = '') {
		$this->spreadsheet = $spreadsheet;
		$this->worksheet = $worksheet;
	}

	public function settitleSpreadsheet($title) {
		$this->spreadsheet = $title;
	}

	public function settitleWorksheet($title) {
		$this->worksheet = $title;
	}

	public static function authUrl() {
		$home_url = home_url();
		$callback = admin_url('admin.php?page=qmn_addons&tab=google-sheet-connector');
		$auth_url = 'https://quizandsurveymaster.com/gshp/auth.php';
		$auth_url .= '?domain='. urlencode($home_url);
		$auth_url .= '&callback='. urlencode($callback);
		return $auth_url;
	}

	public function auth() {
		$client = new Google_Client();
		$client->setClientId(QSM_Google_Sheets::clientId);
		$client->setClientSecret(QSM_Google_Sheets::clientSecret);
		$client->setAccessType("offline");

		$authCode = get_option('gs_accesscode');
		$tokenData = json_decode(get_option('gs_token'), true);
		$client->setAccessToken($tokenData);
		if ($client->isAccessTokenExpired()) {
			if ($client->getRefreshToken()) {
				$accessToken = $client->fetchAccessTokenWithRefreshToken($client->getRefreshToken());
			} else {
				$accessToken = $client->fetchAccessTokenWithAuthCode($authCode);
				if (array_key_exists('error', $accessToken)) {
					throw new Exception(join(', ', $accessToken));
				}
			}
			$tokenData = array_merge($tokenData, $accessToken);
			QSM_Google_Sheets::updateToken($tokenData);
		}
		$this->client = $client;
		return $client;
	}

	public static function updateToken($tokenData) {
		$created = $tokenData['created'];
		$tokenData['expire'] = $created + ($tokenData['expires_in'] - 30);
		try {
			$tokenJson = json_encode($tokenData);
			update_option('gs_token', $tokenJson);
		} catch (Exception $e) {
			error_log("Token write fail! - " . $e->getMessage());
		}
	}

	public function add_row($data, $header = array(), $results = array()) {
		try {
			$client = $this->auth();
			$service = new SpreadsheetSnippets($client);
			$spreadsheetId = $this->spreadsheet;
			$range = $this->worksheet;
			/**
			 * Checks wheter existing header is in sync with current header
			 */
			$header = $service->checkHeader($spreadsheetId, $range, $header, $results);
			/**
			 * Add/update headers of worksheet
			 */
			if (!empty($header)) {
				$service->updateValues($spreadsheetId, "{$this->worksheet}!A1", array(array_keys($header)));
			}
			/**
			 * Insert Data into Sheet
			 */
			if (!empty($header)) {
				$service->appendValues($spreadsheetId, $range, array(array_values($header)));
			}
		} catch (Exception $e) {
			
			error_log("Error - QSMGoogleSheetConnector - " . $e->getMessage());
		}
	}

}

class SpreadsheetSnippets {

	public $service;

	public function __construct($client) {
		$this->service = new Google_Service_Sheets($client);
	}

	public function init() {
		
	}

	public function checkHeader($spreadsheetId, $range, $header, $results = array()){
		$params = [
			'majorDimension' => 'ROWS'
		];
		$result = $this->service->spreadsheets_values->get($spreadsheetId, "{$range}!A1:ZZ1", $params);
		$google_sheets_header = $result['values'][0];
		if ( ! empty( $google_sheets_header ) ) {
			$header_keys			 = array_keys( $header );
			/**
			 * Prepare Contact Fields for Mapping
			 */
			$custom_contact_array	 = array();
			if ( ! empty( $results['contact'] ) ) {
				foreach ( $results['contact'] as $contact ) {
					$custom_contact_array[$contact['label']] = $contact['value'];
					if ( ! empty( $contact['use'] ) ) {
						switch ( $contact['use'] ) {
							case 'name':
								$custom_contact_array['name']		 = $contact['value'];
								break;
							case 'email':
								$custom_contact_array['email']		 = $contact['value'];
								break;
							case 'comp':
								$custom_contact_array['business']	 = $contact['value'];
								break;
							case 'phone':
								$custom_contact_array['phone']		 = $contact['value'];
								break;
							default:
								break;
						}
					}
				}
			}
			/**
			 * Setup New Data Based on Google Sheet Columns.
			 */
			$new_data = array();
			foreach ( $google_sheets_header as $key => $title ) {
				if ( in_array( $title, $header_keys, true ) ) {
					$new_data[$title] = $header[$title];
				} else {
					$new_data[$title] = '';
					if ( in_array( $title, array( 'name', 'email', 'phone', 'business' ) ) && isset( $custom_contact_array[$title] ) ) {
						$new_data[$title] = $custom_contact_array[$title];
					}
				}
			}
			/**
			 * Add Missing Columns
			 */
			$diff = array_diff( $header_keys, $google_sheets_header );
			if ( ! empty( $diff ) ) {
				foreach ( $diff as $key ) {
					$new_data[$key] = $header[$key];
				}
			}
			return $new_data;
		}
		return $header;
	}

	public function updateValues($spreadsheetId, $range, $_values = array(), $valueInputOption = "RAW") {
		$service = $this->service;
		// [START sheets_update_values]
		$values = [
			[
			// Cell values ...
			],
			// Additional rows ...
		];
		// [START_EXCLUDE silent]
		$values = $_values;
		// [END_EXCLUDE]
		$body = new Google_Service_Sheets_ValueRange([
			'values' => $values
		]);
		$valueInputOption = apply_filters( 'qsm_google_sheet_valueInputOption', $valueInputOption );
		$params = [
			'valueInputOption' => $valueInputOption
		];
		$result = $service->spreadsheets_values->update($spreadsheetId, $range, $body, $params);
		//printf("%d cells updated.", $result->getUpdatedCells());
		// [END sheets_update_values]
		return $result;
	}

	public function appendValues($spreadsheetId, $range, $_values = array(), $valueInputOption = "RAW") {
		$service = $this->service;
		// [START sheets_append_values]
		$values = [
			[
			// Cell values ...
			],
			// Additional rows ...
		];
		// [START_EXCLUDE silent]
		$values = $_values;
		// [END_EXCLUDE]
		$body = new Google_Service_Sheets_ValueRange([
			'values' => $values
		]);
		$valueInputOption = apply_filters( 'qsm_google_sheet_valueInputOption', $valueInputOption );
		$params = [
			'valueInputOption' => $valueInputOption
		];
		$result = $service->spreadsheets_values->append($spreadsheetId, $range, $body, $params);
		//printf("%d cells appended.", $result->getUpdates()->getUpdatedCells());
		// [END sheets_append_values]
		return $result;
	}

}